const { Telegraf, Markup } = require('telegraf');
const fs = require('fs');
const path = require('path');
const axios = require('axios');
const FormData = require('form-data');
const config = require('./config');

const bot = new Telegraf(config.TELEGRAM_BOT_TOKEN);

// Load data premium
let premiumData = {};
try {
    premiumData = require('./premium.json');
} catch (e) {
    premiumData = {};
    savePremiumData();
}

// Fungsi untuk menyimpan data premium
function savePremiumData() {
    fs.writeFileSync('./premium.json', JSON.stringify(premiumData, null, 2));
}

// Fungsi untuk memeriksa status premium user
function isPremium(userId) {
    if (config.OWNER_IDS.includes(userId)) return true;
    if (!premiumData[userId]) return false;
    
    const premiumUntil = new Date(premiumData[userId].premiumUntil);
    return premiumUntil > new Date();
}

// Fungsi untuk memeriksa apakah user sudah join channel dan group
async function checkMembership(ctx) {
    try {
        const channelMember = await ctx.telegram.getChatMember(config.CHANNEL_USERNAME, ctx.from.id);
        const groupMember = await ctx.telegram.getChatMember(config.GROUP_USERNAME, ctx.from.id);
        
        return channelMember.status !== 'left' && groupMember.status !== 'left';
    } catch (error) {
        console.error('Error checking membership:', error);
        return false;
    }
}

// Middleware untuk memeriksa premium status
bot.use(async (ctx, next) => {
    if (ctx.message && ctx.message.text) {
        const text = ctx.message.text;
        if (text.startsWith('/deployvercel') || text.startsWith('/deploynetlify')) {
            if (!isPremium(ctx.from.id)) {
                const hasMembership = await checkMembership(ctx);
                
                if (hasMembership) {
                    // Berikan premium gratis
                    const premiumUntil = new Date();
                    premiumUntil.setDate(premiumUntil.getDate() + config.PREMIUM_DAYS);
                    
                    premiumData[ctx.from.id] = {
                        premiumUntil: premiumUntil.toISOString(),
                        grantedBy: 'system',
                        grantedAt: new Date().toISOString()
                    };
                    savePremiumData();
                    
                    await ctx.reply(`🎉 Selamat! Anda mendapatkan akses premium gratis selama ${config.PREMIUM_DAYS} hari karena telah bergabung dengan channel dan group kami.`);
                } else {
                    await ctx.reply(
                        `❌ Fitur ini hanya untuk pengguna premium. Bergabunglah dengan channel dan group kami untuk mendapatkan akses premium gratis!\n\n` +
                        `Channel: ${config.CHANNEL_URL}\n` +
                        `Group: ${config.GROUP_URL}`,
                        Markup.inlineKeyboard([
                            Markup.button.url('Channel', config.CHANNEL_URL),
                            Markup.button.url('Group', config.GROUP_URL)
                        ])
                    );
                    return;
                }
            }
        }
    }
    next();
});

// Command start
bot.start(async (ctx) => {
    const welcomeMessage = `🤖 Bot Deploy Website\n\n` +
        `Saya dapat membantu Anda mendeploy website secara otomatis ke Vercel atau Netlify.\n\n` +
        `Cara penggunaan:\n` +
        `1. Kirim file HTML ke bot\n` +
        `2. Reply file tersebut dengan perintah:\n` +
        `   /deployvercel namadomain\n` +
        `   atau\n` +
        `   /deploynetlify namadomain\n\n` +
        `Fitur hanya tersedia untuk pengguna premium.`;
    
    await ctx.replyWithPhoto(
        { url: 'https://f.top4top.io/p_3496a9bvj1.jpg?text=Bot+Deploy+Website' },
        {
            caption: welcomeMessage,
            parse_mode: 'HTML',
            ...Markup.inlineKeyboard([
                Markup.button.url('Developer', `https://t.me/${config.DEVELOPER_USERNAME}`),
                Markup.button.url('Channel', config.CHANNEL_URL),
                Markup.button.url('Group', config.GROUP_URL)
            ])
        }
    );
});

// Command untuk deploy ke Vercel
bot.command('deployvercel', async (ctx) => {
    if (!ctx.message.reply_to_message || !ctx.message.reply_to_message.document) {
        return ctx.reply('❌ Harap reply pesan yang berisi file HTML dengan perintah ini.');
    }
    
    const args = ctx.message.text.split(' ');
    if (args.length < 2) {
        return ctx.reply('❌ Harap sertakan nama domain. Contoh: /deployvercel mywebsite');
    }
    
    const domainName = args[1];
    const fileId = ctx.message.reply_to_message.document.file_id;
    
    try {
        await ctx.reply('⏳ Sedang memproses deploy ke Vercel...');
        
        // Download file
        const fileLink = await ctx.telegram.getFileLink(fileId);
        const response = await axios.get(fileLink, { responseType: 'stream' });
        
        // Buat repository di GitHub
        const repoName = `vercel-${domainName}-${Date.now()}`;
        const githubResponse = await axios.post(
            'https://api.github.com/user/repos',
            {
                name: repoName,
                auto_init: false,
                private: false
            },
            {
                headers: {
                    Authorization: `token ${config.GITHUB_TOKEN}`,
                    Accept: 'application/vnd.github.v3+json'
                }
            }
        );
        
        // Upload file ke GitHub
        const content = await streamToBuffer(response.data);
        const contentBase64 = content.toString('base64');
        
        await axios.put(
            `https://api.github.com/repos/${config.GITHUB_USERNAME}/${repoName}/contents/index.html`,
            {
                message: 'Add index.html',
                content: contentBase64
            },
            {
                headers: {
                    Authorization: `token ${config.GITHUB_TOKEN}`,
                    Accept: 'application/vnd.github.v3+json'
                }
            }
        );
        
        // Deploy ke Vercel
        const vercelResponse = await axios.post(
            'https://api.vercel.com/v13/deployments',
            {
                name: repoName,
                gitSource: {
                    type: 'github',
                    repoId: githubResponse.data.id,
                    ref: 'main'
                },
                target: 'production'
            },
            {
                headers: {
                    Authorization: `Bearer ${config.VERCEL_API_TOKEN}`,
                    'Content-Type': 'application/json'
                }
            }
        );
        
        await ctx.reply(`✅ Website berhasil di-deploy ke Vercel!\n\n` +
            `🌐 Domain: https://${domainName}.vercel.app\n` +
            `📊 Status: ${vercelResponse.data.status}`);
        
    } catch (error) {
        console.error('Error deploying to Vercel:', error.response?.data || error.message);
        await ctx.reply('❌ Gagal mendeploy ke Vercel. Silakan coba lagi nanti.');
    }
});

// Command untuk deploy ke Netlify
bot.command('deploynetlify', async (ctx) => {
    if (!ctx.message.reply_to_message || !ctx.message.reply_to_message.document) {
        return ctx.reply('❌ Harap reply pesan yang berisi file HTML dengan perintah ini.');
    }
    
    const args = ctx.message.text.split(' ');
    if (args.length < 2) {
        return ctx.reply('❌ Harap sertakan nama domain. Contoh: /deploynetlify mywebsite');
    }
    
    const domainName = args[1];
    const fileId = ctx.message.reply_to_message.document.file_id;
    
    try {
        await ctx.reply('⏳ Sedang memproses deploy ke Netlify...');
        
        // Download file
        const fileLink = await ctx.telegram.getFileLink(fileId);
        const response = await axios.get(fileLink, { responseType: 'stream' });
        
        // Buat form data untuk Netlify
        const formData = new FormData();
        formData.append('file', response.data, { filename: 'index.html' });
        
        // Deploy ke Netlify
        const netlifyResponse = await axios.post(
            'https://api.netlify.com/api/v1/sites',
            formData,
            {
                headers: {
                    Authorization: `Bearer ${config.NETLIFY_API_TOKEN}`,
                    ...formData.getHeaders()
                }
            }
        );
        
        // Update site name jika diperlukan
        if (domainName !== netlifyResponse.data.subdomain) {
            await axios.patch(
                `https://api.netlify.com/api/v1/sites/${netlifyResponse.data.id}`,
                {
                    name: domainName
                },
                {
                    headers: {
                        Authorization: `Bearer ${config.NETLIFY_API_TOKEN}`
                    }
                }
            );
        }
        
        await ctx.reply(`✅ Website berhasil di-deploy ke Netlify!\n\n` +
            `🌐 Domain: https://${netlifyResponse.data.subdomain}.netlify.app\n` +
            `📊 Status: ${netlifyResponse.data.state}`);
        
    } catch (error) {
        console.error('Error deploying to Netlify:', error.response?.data || error.message);
        await ctx.reply('❌ Gagal mendeploy ke Netlify. Silakan coba lagi nanti.');
    }
});

// Command untuk menambah premium (owner only)
bot.command('addpremium', async (ctx) => {
    if (!config.OWNER_IDS.includes(ctx.from.id)) {
        return ctx.reply('❌ Anda tidak memiliki akses untuk perintah ini.');
    }
    
    const args = ctx.message.text.split(' ');
    if (args.length < 3) {
        return ctx.reply('❌ Format: /addpremium <user_id> <days>');
    }
    
    const userId = parseInt(args[1]);
    const days = parseInt(args[2]);
    
    if (isNaN(userId) || isNaN(days)) {
        return ctx.reply('❌ User ID dan hari harus berupa angka.');
    }
    
    const premiumUntil = new Date();
    premiumUntil.setDate(premiumUntil.getDate() + days);
    
    premiumData[userId] = {
        premiumUntil: premiumUntil.toISOString(),
        grantedBy: ctx.from.id,
        grantedAt: new Date().toISOString()
    };
    
    savePremiumData();
    
    await ctx.reply(`✅ User ${userId} telah ditambahkan sebagai premium selama ${days} hari.`);
    
    try {
        await ctx.telegram.sendMessage(userId, `🎉 Anda telah mendapatkan akses premium selama ${days} hari!`);
    } catch (error) {
        console.error('Cannot send message to user:', error);
    }
});

// Command untuk menghapus premium (owner only)
bot.command('delpremium', async (ctx) => {
    if (!config.OWNER_IDS.includes(ctx.from.id)) {
        return ctx.reply('❌ Anda tidak memiliki akses untuk perintah ini.');
    }
    
    const args = ctx.message.text.split(' ');
    if (args.length < 2) {
        return ctx.reply('❌ Format: /delpremium <user_id>');
    }
    
    const userId = parseInt(args[1]);
    
    if (isNaN(userId)) {
        return ctx.reply('❌ User ID harus berupa angka.');
    }
    
    if (premiumData[userId]) {
        delete premiumData[userId];
        savePremiumData();
        await ctx.reply(`✅ User ${userId} telah dihapus dari premium.`);
    } else {
        await ctx.reply(`❌ User ${userId} tidak ditemukan dalam daftar premium.`);
    }
});

// Command untuk broadcast (owner only)
bot.command('broadcastuser', async (ctx) => {
    if (!config.OWNER_IDS.includes(ctx.from.id)) {
        return ctx.reply('❌ Anda tidak memiliki akses untuk perintah ini.');
    }
    
    if (!ctx.message.reply_to_message) {
        return ctx.reply('❌ Harap reply pesan yang ingin di-broadcast.');
    }
    
    const message = ctx.message.reply_to_message;
    const userIds = Object.keys(premiumData);
    
    await ctx.reply(`⏳ Mengirim broadcast ke ${userIds.length} pengguna...`);
    
    let successCount = 0;
    let failCount = 0;
    
    for (const userId of userIds) {
        try {
            await ctx.telegram.copyMessage(userId, ctx.chat.id, message.message_id);
            successCount++;
        } catch (error) {
            console.error(`Cannot send message to user ${userId}:`, error);
            failCount++;
        }
        
        // Delay untuk menghindari rate limiting
        await new Promise(resolve => setTimeout(resolve, 100));
    }
    
    await ctx.reply(`✅ Broadcast selesai!\nBerhasil: ${successCount}\nGagal: ${failCount}`);
});

// Helper function untuk mengconvert stream ke buffer
function streamToBuffer(stream) {
    return new Promise((resolve, reject) => {
        const chunks = [];
        stream.on('data', chunk => chunks.push(chunk));
        stream.on('error', reject);
        stream.on('end', () => resolve(Buffer.concat(chunks)));
    });
}

// Handle errors
bot.catch((err, ctx) => {
    console.error(`Error for ${ctx.updateType}:`, err);
});

// Start bot
bot.launch().then(() => {
    console.log('Bot started successfully');
});

// Enable graceful stop
process.once('SIGINT', () => bot.stop('SIGINT'));
process.once('SIGTERM', () => bot.stop('SIGTERM'));